class Member:
    """Represents a Weverse Member.

    .. container:: operations

        .. describe:: x == y

            Checks if two members are equal.

        .. describe:: x != y

            Checks if two members are not equal.

        .. describe:: hash(x)

            Returns the member's hash.

        .. describe:: str(x)

            Returns the member's name.

    Attributes
    ----------
    data: :class:`str`
        The raw data directly taken from the response generated by Weverse's API.
    id: :class:`str`
        The ID of the member.
    community_id: :class:`int`
        The community ID of the community the member belongs to. This
        can be used to fetch the actual :class:`sunverse.objects.Community`
        object if needed using the :method:`fetch_community` method.
    joined: :class:`bool`
        Whether the member has joined the community.
    name: :class:`str`
        The name of the member.
    image_url: Optional[:class:`str`]
        The URL of the profile image of the member, if any.
    banner_url: Optional[:class:`str`]
        The URL of the profile banner of the member, if any.
    profile_comment: Optional[:class:`str`]
        The profile comment of the member, if any.
    follower_count: :class:`int`
        The number of users following the member.
    member_join_status: :class:`str`
        The join status of the member.
    profile_type: :class:`str`
        The profile type of the member.
    hidden: :class:`bool`
        UNDETERMINED FUNCTIONALITY
    blinded: :class:`bool`
        UNDETERMINED FUNCTIONALITY
    has_membership :class:`bool`
        Whether the member has a paid subscription to the community.
    has_official_mark: :class:`bool`
        Whether the member is an official Weverse account.
    joined_date: :class:`int`
        The date the member joined the community on, in epoch.
    """

    __slots__ = (
        "data",
        "id",
        "community_id",
        "joined",
        "name",
        "image_url",
        "banner_url",
        "profile_comment",
        "follower_count",
        "member_join_status",
        "profile_type",
        "hidden",
        "blinded",
        "has_membership",
        "has_official_mark",
        "joined_date",
    )

    def __init__(self, data: dict):
        self.data: dict = data
        self.id: str = data["memberId"]
        self.community_id: int = data["communityId"]
        self.joined: bool = data["joined"]
        self.name: str = data["profileName"]
        self.image_url: str | None = data.get("profileImageUrl")
        self.banner_url: str | None = data.get("profileCoverImageUrl")
        self.profile_comment: str | None = data.get("profileComment")
        self.follower_count: int = (
            data["followCount"]["followerCount"] if data.get("followCount") else None
        )
        self.member_join_status: str = data["memberJoinStatus"]
        self.profile_type: str = data["profileType"]
        self.hidden: bool = data["hidden"]
        self.blinded: bool = data["blinded"]
        self.has_membership: bool = data["hasMembership"]
        self.has_official_mark: bool = data["hasOfficialMark"]
        self.joined_date: int = data["joinedDate"]

    def __eq__(self, other):
        if isinstance(other, self.__class__):
            return self.id == other.id

        raise NotImplementedError

    def __repr__(self):
        return f"Member member_id={self.id}, name={self.name}"

    def __str__(self):
        return self.name

    def __hash__(self):
        return hash(self.id)
